document.addEventListener('DOMContentLoaded', () => {
    // --- Canvas Particle Background ---
    const canvas = document.getElementById('canvas-bg');
    const ctx = canvas.getContext('2d');
    let width, height;
    let particles = [];

    function resize() {
        width = window.innerWidth;
        height = window.innerHeight;
        canvas.width = width;
        canvas.height = height;
    }

    class Particle {
        constructor() {
            this.x = Math.random() * width;
            this.y = Math.random() * height;
            this.vx = (Math.random() - 0.5) * 0.5;
            this.vy = (Math.random() - 0.5) * 0.5;
            this.size = Math.random() * 2 + 1;
            // Tricolor palette
            const colors = ['rgba(255, 153, 51, 0.5)', 'rgba(255, 255, 255, 0.5)', 'rgba(19, 136, 8, 0.5)'];
            this.color = colors[Math.floor(Math.random() * colors.length)];
        }

        update() {
            this.x += this.vx;
            this.y += this.vy;

            if (this.x < 0 || this.x > width) this.vx *= -1;
            if (this.y < 0 || this.y > height) this.vy *= -1;
        }

        draw() {
            ctx.beginPath();
            ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2);
            ctx.fillStyle = this.color;
            ctx.fill();
        }
    }

    function initParticles() {
        particles = [];
        const particleCount = Math.min(width * 0.1, 150); // Responsive count
        for (let i = 0; i < particleCount; i++) {
            particles.push(new Particle());
        }
    }

    function animate() {
        ctx.clearRect(0, 0, width, height);
        particles.forEach(p => {
            p.update();
            p.draw();
        });

        // Draw connections
        particles.forEach((a, index) => {
            for (let j = index + 1; j < particles.length; j++) {
                const b = particles[j];
                const dx = a.x - b.x;
                const dy = a.y - b.y;
                const distance = Math.sqrt(dx * dx + dy * dy);

                if (distance < 100) {
                    ctx.beginPath();
                    ctx.strokeStyle = `rgba(255, 255, 255, ${0.1 - distance / 1000})`;
                    ctx.lineWidth = 0.5;
                    ctx.moveTo(a.x, a.y);
                    ctx.lineTo(b.x, b.y);
                    ctx.stroke();
                }
            }
        });

        requestAnimationFrame(animate);
    }

    window.addEventListener('resize', () => {
        resize();
        initParticles();
    });

    resize();
    initParticles();
    animate();

    // --- Interactivity ---

    // 1. Send Wishes (Confetti + Share)
    const wishBtn = document.getElementById('wish-btn');
    const toast = document.getElementById('toast');

    wishBtn.addEventListener('click', async () => {
        // Tricolor Confetti
        const end = Date.now() + 3000;
        const colors = ['#FF9933', '#ffffff', '#138808'];

        (function frame() {
            confetti({
                particleCount: 3,
                angle: 60,
                spread: 55,
                origin: { x: 0 },
                colors: colors
            });
            confetti({
                particleCount: 3,
                angle: 120,
                spread: 55,
                origin: { x: 1 },
                colors: colors
            });

            if (Date.now() < end) {
                requestAnimationFrame(frame);
            }
        }());

        // Sharing Logic
        const shareData = {
            title: 'Happy 77th Republic Day!',
            text: 'Celebrating the Spirit of Sovereignty, Democracy, and Unity. Join me in hoisting the flag!',
            url: window.location.href
        };

        if (navigator.share) {
            try {
                await navigator.share(shareData);
            } catch (err) {
                console.log('Error sharing:', err);
            }
        } else {
            // Fallback: Copy to Clipboard
            try {
                await navigator.clipboard.writeText(window.location.href);
                showToast();
            } catch (err) {
                console.error('Failed to copy: ', err);
            }
        }
    });

    function showToast() {
        toast.classList.add('show');
        toast.classList.remove('hidden');
        setTimeout(() => {
            toast.classList.remove('show');
            setTimeout(() => {
                toast.classList.add('hidden');
            }, 500); // Wait for transition
        }, 3000);
    }

    // 2. Hoist Flag
    const hoistBtn = document.getElementById('hoist-btn');
    const flagOverlay = document.getElementById('flag-overlay');
    const closeFlagBtn = document.getElementById('close-flag');
    const anthem = document.getElementById('anthem');

    hoistBtn.addEventListener('click', () => {
        flagOverlay.classList.remove('hidden');
        // Reset flag position just in case
        const flag = document.querySelector('.flag');
        flag.classList.remove('hoist-active');

        // Small delay to allow display:block to apply before opacity transition
        setTimeout(() => {
            flagOverlay.classList.add('active');

            // Trigger Hoisting Animation
            requestAnimationFrame(() => {
                flag.classList.add('hoist-active');
            });
        }, 10);

        // Play anthem
        anthem.currentTime = 0;
        anthem.volume = 0.5;
        anthem.play().catch(e => console.log("Audio play failed (user interaction needed):", e));

        // Trigger confetti burst
        confetti({
            particleCount: 150,
            spread: 70,
            origin: { y: 0.6 },
            colors: ['#FF9933', '#ffffff', '#138808']
        });
    });

    closeFlagBtn.addEventListener('click', () => {
        flagOverlay.classList.remove('active');
        anthem.pause();

        setTimeout(() => {
            flagOverlay.classList.add('hidden');
            // Reset Flag Animation
            document.querySelector('.flag').classList.remove('hoist-active');
        }, 500);
    });
});
